//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	upper and lower bound inflow estimates
// Objective: 	Producing upper and lower bound inflow estimates. The lower
//				bound includes inheritances where the recipient also reported
//				the death of a close family member (excluding spouses and
//				children). The upper bound estimate excludes inheritances likely
//				to have come from a spouse.
//
// Created: 	25/05/2021
// Modified: 	15/11/2021
//==============================================================================

**************************
*** Importing the Data ***
**************************

clear

cd "XXXX"	// set the content directory here

use HILDA_restricted_combined_rescaled	// importing the data

xtset xwaveid wave	// declaring the dataset to be a panel

//	generating a bequest indicator variable
gen bequest = 0
replace bequest = 1 if oifinh == 1

by xwaveid, sort: egen num_bequests = sum(bequest)	// calculating the number of bequests by wave

*************************
*** Cleaning the data ***
*************************

drop if num_bequests == 0	// dropping observations that did not report the receipt of a bequest
drop if scmatch != 1	// dropping observations that did not match up with a self-completion questionnaire (SCQ)

********************************
*** Generating key variables ***
********************************

//	generating an indicator variable of whether a close relative (excluding spouses and children) has died in the past three years
gen death_of_relative = 0
replace death_of_relative = 1 if ledrl == 2
replace death_of_relative = 1 if L.ledrl == 2
replace death_of_relative = 1 if L2.ledrl == 2
replace death_of_relative = 1 if L3.ledrl == 2
replace death_of_relative = 1 if F.ledrl == 2

//	generating an indicator variable of whether a spouse or child has died in the past three years
gen death_of_spouse_child = 0
replace death_of_spouse_child = 1 if ledsc == 2
replace death_of_spouse_child = 1 if L.ledsc == 2
replace death_of_spouse_child = 1 if L2.ledsc == 2
replace death_of_spouse_child = 1 if L3.ledsc == 2
replace death_of_spouse_child = 1 if F.ledsc == 2

//	generating an indicator variable equal to one if any person close to the repsondent has died in the previous three years
gen death = 0
replace death = 1 if (death_of_relative == 1 | death_of_spouse_child == 1)

//	generating a 'who died' variable with all the possible combinations of relative, spouse or child and friend deaths
gen who_died = 0
replace who_died = 1 if death_of_relative == 1 & death_of_spouse_child == 0
replace who_died = 2 if death_of_relative == 0 & death_of_spouse_child == 1
replace who_died = 3 if death_of_relative == 1 & death_of_spouse_child == 1

//	adding labels to the 'who died' variable
label define who_died_label 0 "No one" 1 "Only a close relative" 2 "Only a spouse or child" 3 "A close relative and a spouse or child"
label values who_died who_died_label	// adding the labels to the 'who died' variable

replace oifinha = 0 if oifinha < 0	// replacing negative inheritances as zero

gen wt_inheritance = oifinha * hhwtsc	// weighting inheritances by the self-completion questionnaire weights

****************
*** Analysis ***
****************

by year who_died, sort: egen num_inheritances = count(oifinh) if oifinh == 1	// calculating the number of inheritances by year and who died

by year who_died, sort: egen sum_inheritances = sum(wt_inheritance) if oifinh == 1	// calculating the sum of inheritances by year and who died

collapse num_inheritances sum_inheritances, by(year who_died)	// collapsing the key variables by year and who died

reshape wide num_inheritances sum_inheritances, i(year) j(who_died)	// reshaping the dataset

* renaming variables
rename num_inheritances0 no_one_num
rename sum_inheritances0 no_one_sum
rename num_inheritances1 family_num
rename sum_inheritances1 family_sum
rename num_inheritances2 spouse_child_num
rename sum_inheritances2 spouse_child_sum
rename num_inheritances3 both_num
rename sum_inheritances3 both_sum

save inheritances_by_who_died.dta, replace	// saving the dataset

****************************
*** Upper bound estimate ***
****************************

clear	// clearing the workspace

use HILDA_restricted_combined_rescaled	// opening the dataset

merge m:1 xwaveid wave using first_estate_bequest.dta	// merging in the first estate bequest filter

drop if _merge == 1	// dropping non-responding persons
drop _merge	// dropping the _merge variable

replace oifinha = 0 if oifinha < 0	// replacing negative bequest values as zero
gen bequest = oifinha	// generating a bequest variable
replace bequest = 0 if bequest > 0 & first_estate_bequest == 1	// replacing first estate bequests as zero to exclude them from the analysis

gen wt_inheritance = bequest * hhwtrp	// weighting the bequest variable by the responding-person population level weights

by year, sort: egen num_inheritances = count(wt_inheritance) if wt_inheritance > 0	// calculating the number of inheritances by year

by year, sort: egen sum_inheritances = sum(wt_inheritance) if wt_inheritance > 0	// calculating the sum of inheritances by year

collapse num_inheritances sum_inheritances, by(year)	// collapsing the key variables by year

save inheritances_excluding_HH.dta, replace	// saving the dataset

***************************************************************
*** Merging datasets and converting to constant 2018 prices ***
***************************************************************

merge 1:1 year using inheritances_by_who_died.dta, nogen	// merging in lower bound estimates

keep year sum_inheritances family_sum	// keeping key variables

* renaming upper and lower bound estimates
rename sum_inheritances upper_bound
rename family_sum lower_bound

merge 1:1 year using CPI_FY_2018.dta, nogen	// merging in financial-year CPI data with 2018 as the base year

* converting to constant 2018 dollars
replace upper_bound = upper_bound * (100 / cpi)
replace lower_bound = lower_bound * (100 / cpi)

save upper_and_lower_bound_inflow_estimates.dta, replace	// saving the dataset